#ifndef AHLGREN_PARAMETERS
#define AHLGREN_PARAMETERS

#include <array>
#include <string>
#include <exception>

#include "global.h" // verbosity
#include "functor_tbl.h"


namespace lp {

	// Exception class
	struct invalid_name : public std::exception { 
		invalid_name(const char* name) : std::exception(name) {}
		invalid_name(const string& s) : std::exception(s.c_str()) {}
	};

	class parameters {
	public:
		// Parameters
		enum index {
			qdepth, qres, qwarnings, /*filter_subs,*/ require_defs, require_files, // queries
			bsize, csize, memo_queries, memo_types, memoize, recursion_penalty, inflate, noise, // induction
			enforce_io_spec, functional, // induction 
			evalfn, evalpartial, terminate, mincover, solution_if, // induction (eval/termination)
			reorder_negs, reorder_negs_factor, // Special induction features
			nodes, max_nodes, max_bottom, max_instructions, max_examples, time_limit, // induction limits
			pos_order, biterations, splitvars, // bottom clause
			detect_dependent, detect_reflexivity, detect_symmetry, /*detect_diff,*/ // pre-processing
			compress, flatten, validate, post_detect, post_detect_samples, // post-processing
			// SAT settings
			sat_solver, dpllass, dpll_strategy, vsids_decay, dpll_cllsize, dpll_cll, dpll_learn,
			dpll_cll_max_occurs, dpll_cll_clause_length,
			// SAT mode constraints
			chain_io, chain_return, weak_chain_io, chain_arguments, 
			// SAT pruning constraints
			prune_inconsistent, prune_consistent, prune_visit,
			conditional_constraints, unconditional_constraints,
			// QG/GA
			popsize, crossover, mutation, tournament, shuffle, qg_memo, ga_memo,
			// Printing and updating
			print_labels, update_labels,
			graph,
			// Pos only learning
			posonly, psamples,
			param_size // there's no way to automatically get enum size
		};

		// Array type
		typedef std::array<data,param_size> paramv;

		// Default Constructor: set all parameters to defaults
		parameters() { reset(); }

		// Access by index
		data& operator[](index i) { return pa[i]; }
		const data& operator[](index i) const { return pa[i]; }
		// Access by shortname char
		const data& operator[](char c) const { return const_cast<parameters&>(*this).operator[](c); }
		data& operator[](char);
		// Access by full name
		const data& operator[](const string& s) const { return const_cast<parameters&>(*this).operator[](s); }
		data& operator[](const string&);

		// Force integer type: tries: actual int value -> default int value -> 0
		int force_int(index i) const;
		double force_float(index i) const;
		bool is_set(index i) const;
		bool is_unset(index i) const;

		// Get default value for a parameter
		static const data& get_default(index i) { return def_pa[i]; }

		void reset() { pa = def_pa; }

		static paramv init_defaults();
	protected:
		paramv pa;
		static paramv def_pa;
	};


	inline parameters::paramv parameters::init_defaults()
	{
		paramv p;

		// Query
		p[qdepth]		= (long long) 30; // Max depth of resolution
		p[qres]			= (long long) 10000; // Max number of resolutions
		p[csize]		= (long long) 4; // max no of atoms in body of hypothesis
		p[require_defs] = (long long) 1; // warn/exit with error if undefined predicate is called
		p[require_files] = (long long) 1; // exit with error if undefined predicate is called
		// Bottom Clause Construction
		p[pos_order]	= "sequential"; // example ordering: sequential, shuffle, complexity
		p[bsize]		= (long long) 10000; // bottom clause max body size (cannot be set to more than about 37000 due to recursion)
		p[biterations]	= (long long) 3; // max number of iterations in bottom clause construction
		p[memo_queries] = (long long) 1; // memoize queries during bottom construction
		p[memo_types] = (long long) 16; // memoize types up to size during bottom construction
		p[splitvars]	= (long long) 0; // variable split during bottom clause construction
		p[detect_dependent]	= (long long) 0; // detect non-dummy variables (0=no, 1=strict, 2=relaxed)
		p[detect_reflexivity]	= (long long) 0; // detect reflexive modeh predicates
		p[detect_symmetry]	= (long long) 0; // detect symmetric modeh predicates
		p[post_detect] = (long long) 0; // post-detect reflexivity and symmetry
		p[post_detect_samples] = (long long) 100; // post-detect sample size
		//p[detect_diff]	= (long long) 0; // detect list length differences
		// General Induction
		p[nodes]		= (long long) 1000; // max number of samples during search
		p[max_nodes]	= (long long) numeric_limits<int>::max(); // total max number of samples during all searches
		p[max_bottom]	= (long long) numeric_limits<int>::max(); // max number of bottom clauses constructed
		p[max_instructions]	= (long long) numeric_limits<int>::max(); // max number of WAM instructions executed (per call)
		p[max_examples]		= (long long) numeric_limits<int>::max(); // max number of examples to evaluate
		p[time_limit]	= (double) numeric_limits<double>::max(); // max number of seconds
		p[mincover]		= (long long) 1; // minimal number of examples covered required for induction
		p[solution_if]	= (double) 1.0; // terminate current search if candidate covers this percentage of positive examples
		p[compress]		= (long long) 1; // compress: 0=nothing, 1=yes, 2=filtered compression
		p[flatten]		= (long long) 1; // flatten clauses after induction (bool)
		p[validate]		= (long long) 0; // validate (1=leave one out, k = k-fold validation)
		p[inflate]		= (long long) 100; // percentage inflation required for induction
		p[qwarnings]	= (long long) 1; // enable/disable query warnings
		p[noise]		= (long long) 0; // number of negative examples accepted
		p[enforce_io_spec] = (long long) 1; // candidates are required to conform to mode declarations
		p[functional] = (long long) 1; // use functional constraints
		p[evalfn]		= "auto"; // fitness evaluation function
		p[evalpartial]	= "auto"; // partial fitness evaluation?
		p[terminate]	= "auto"; // set search termination (can have multiple levels)
		p[reorder_negs]	= (long long) 0; // Continuously reorder negative examples for faster falsification: 1 = instruction based, 2 = time based
		p[reorder_negs_factor]	= (double) 1.0; // c in computation of 1.0/(1.0 + c * time), 0 means that query time doesn't matter
		p[memoize]		= (long long) 0; // memoize during search (only used by some algorithms, e.g. A*)
		p[recursion_penalty] = (long long) 0; // penalize candidates with recursive body literals (by lowering fitness)
		// NrSample
		p[sat_solver]	= "dpll"; // which SAT solver to use ("dpll", "dual_horn")
		p[dpllass]		= (long long) 50000; // max number of assignments in SAT solver
		p[dpll_cll]		= (long long) std::numeric_limits<int>::max(); // max number of clauses learnt
		p[dpll_cllsize]	= (long long) std::numeric_limits<int>::max(); // max size of clauses learnt
		//p[dpll_backjump] = (long long) 0; // use non-chronological backtracking?
		p[dpll_learn]	= (long long) 1; // stop clause learning at 1uip?
		p[dpll_strategy]= "complexity"; // DPLL strategy, default to "complexity"
		p[dpll_cll_clause_length] = (long long) 1; // use clause learning when clause_length has been reached?
		p[dpll_cll_max_occurs] = (long long) 1; // use clause learning when max_occurs has been reached?
		p[vsids_decay]	= 2.0; // decay rate for VSIDS algorithm, scale] = (long long) 1/decay
		p[chain_io] = (long long) 1; // chain literals as input-output
		p[chain_return] = (long long) 1; // all head outputs must be instantiated
		p[chain_arguments] = (long long) 0; // all head inputs must be used
		p[weak_chain_io] = (long long) 0; // use head-connected constraints
		p[prune_inconsistent] = (long long) 1; // prune up when inconsistent
		p[prune_consistent] = (long long) 1; // prune down when consistent
		p[prune_visit] = (long long) 0; // prune visited solution
		p[conditional_constraints] = (long long) 1; // SAT literal constraints
		p[unconditional_constraints] = (long long) 1; // bottom clause constraints

		// QG/GA
		p[qg_memo]		= (long long) 1; // memoize reductions
		p[ga_memo]		= (long long) 1; // memoize GA fitness evaluations
		p[popsize]		= (long long) 30; // population size
		p[shuffle]		= (long long) 1000; // max attempt to reshuffle
		// p[maxgen]	= (long long) 32; // GA max generations
		p[crossover]	= 0.6; // crossover prob: 60%
		p[mutation]		= 0.01; // mutation prob: 1%
		p[tournament]	= (long long) 2; // tournament size
		// Clause Labels
		p[print_labels]	= (long long) 0; // print clause labels
		p[update_labels]= (long long) 0; // print clause labels
		// Graph
		p[graph]	= (long long) 0; // print search graph
		// Induction
		p[posonly]		= (long long) 0; // learn from positive examples only
		p[psamples]		= (long long) 100; // Use psamples when learning with posonly

		return p;
	}


	inline data& parameters::operator[](char ch)
	{
		switch (ch) {
		case 'h': return pa[qdepth];
		case 'r': return pa[qres];
		case 'b': return pa[bsize];
		case 'c': return pa[csize];
		case 'n': return pa[nodes];
		case 'm': return pa[mincover];
		case 's': return pa[solution_if];
		case 'z': return pa[compress];
		case 'a': return pa[dpllass];
		case 'i': return pa[biterations];
		case 't': return pa[time_limit];
		// pointers to variables declared elsewhere
		case 'p': return pretty_print;
		case 'v': return verbosity;
		default: throw invalid_name(string(1,ch));
		}
	}


	inline data& parameters::operator[](const string& s)
	{
		if (s.length() == 1) {
			// short form
			return (*this)[ s[0] ];
		}
		// long form
		if (s == "query_depth") return pa[qdepth];
		if (s == "resolutions") return pa[qres];
		if (s == "bottom_size") return pa[bsize];
		if (s == "splitvars") return pa[splitvars];
		if (s == "detect_dependent") return pa[detect_dependent];
		if (s == "detect_reflexivity") return pa[detect_reflexivity];
		if (s == "detect_symmetry") return pa[detect_symmetry];
		if (s == "post_detect") return pa[post_detect];
		if (s == "post_detect_samples") return pa[post_detect_samples];
		//if (s == "detect_diff") return pa[detect_diff];
		if (s == "clauselength") return pa[csize];
		if (s == "pos_order") return pa[pos_order];
		if (s == "query_warnings") return pa[qwarnings];
		if (s == "nodes") return pa[nodes];
		if (s == "max_nodes") return pa[max_nodes];
		if (s == "max_bottom") return pa[max_bottom];
		if (s == "max_instructions") return pa[max_instructions];
		if (s == "max_examples") return pa[max_examples];
		if (s == "time_limit") return pa[time_limit];
		if (s == "posonly") return pa[posonly];
		if (s == "psamples") return pa[psamples];
		if (s == "evalfn") return pa[evalfn];
		if (s == "evalpartial") return pa[evalpartial];
		if (s == "terminate") return pa[terminate];
		if (s == "reorder_negs") return pa[reorder_negs];
		if (s == "reorder_negs_factor") return pa[reorder_negs_factor];
		if (s == "min_cover") return pa[mincover];
		if (s == "solution_if") return pa[solution_if];
		if (s == "noise") return pa[noise];
		if (s == "enforce_io_spec") return pa[enforce_io_spec];
		if (s == "functional") return pa[functional];
		if (s == "post_compression") return pa[compress];
		if (s == "flatten") return pa[flatten];
		if (s == "validate") return pa[validate];
		if (s == "sat_solver") return pa[sat_solver];
		if (s == "dpll_assignments") return pa[dpllass];
		if (s == "dpll_cll") return pa[dpll_cll];
		if (s == "dpll_cllsize") return pa[dpll_cllsize];
		if (s == "dpll_learn") return pa[dpll_learn];
		if (s == "dpll_cll_clause_length") return pa[dpll_cll_clause_length];
		if (s == "dpll_cll_max_occurs") return pa[dpll_cll_max_occurs];
		//if (s == "dpll_backjump") return pa[dpll_backjump];
		if (s == "dpll_strategy") return pa[dpll_strategy];
		if (s == "vsids_decay") return pa[vsids_decay];
		if (s == "chain_io") return pa[chain_io];
		if (s == "chain_return") return pa[chain_return];
		if (s == "chain_argument") return pa[chain_arguments];
		if (s == "weak_chain_io") return pa[weak_chain_io];
		if (s == "prune_inconsistent") return pa[prune_inconsistent];
		if (s == "prune_consistent") return pa[prune_consistent];
		if (s == "prune_visit") return pa[prune_visit];
		if (s == "conditional_constraints") return pa[conditional_constraints];
		if (s == "unconditional_constraints") return pa[unconditional_constraints];
		if (s == "inflate") return pa[inflate];
		if (s == "var_depth") return pa[biterations];
		if (s == "memo_queries") return pa[memo_queries];
		if (s == "memo_types") return pa[memo_types];
		if (s == "pop_size") return pa[popsize];
		if (s == "memo") return pa[memoize];
		if (s == "recursion_penalty") return pa[recursion_penalty];
		if (s == "crossover") return pa[crossover];
		if (s == "mutation") return pa[mutation];
		if (s == "tournament") return pa[tournament];
		if (s == "shuffle") return pa[shuffle];
		if (s == "qg_memo") return pa[qg_memo];
		if (s == "ga_memo") return pa[ga_memo];
		if (s == "print_labels") return pa[print_labels];
		if (s == "update_labels") return pa[update_labels];
		if (s == "graph") return pa[graph];
		// if (s == "filter_subs") return pa[filter_subs];
		if (s == "require_definitions") return pa[require_defs];
		if (s == "require_files") return pa[require_files];
		// Pointers to variables declared elsewhere
		if (s == "verbose")	return verbosity;
		if (s == "pretty_print") return pretty_print;
		throw invalid_name(s);
	}


	inline int parameters::force_int(index i) const
	{
		const data& dat = pa[i];
		if (dat.is_int()) {
			return static_cast<int>(dat.get_int());
		} else if (dat.is_atom() && strcmp(dat.get_atom(),"*") == 0) {
			// Default: unlimited
			return numeric_limits<int>::max();
		} else {
			const data& def = get_default(i);
			if (def.is_int()) {
				return static_cast<int>(def.get_int());
			} else {
				return 0;
			}
		}
	}

	inline double parameters::force_float(index i) const
	{
		const data& dat = pa[i];
		if (dat.is_double()) {
			return dat.get_float();
		} else if (dat.is_int()) { // try to convert int to float
			const auto val = dat.get_int();
			return double(val);
		} else {
			const data& def = get_default(i);
			if (def.is_double()) {
				return def.get_float();
			} else {
				return 0;
			}
		}
	}

	inline bool parameters::is_set(index i) const
	{
		const data& dat = pa[i];
		return dat.is_int() && dat.get_int() > 0 
			|| dat.is_atom() && strcmp(dat.get_atom(),"true") == 0
			|| dat.is_double() && dat.get_float() > 0.0;
	}

	inline bool parameters::is_unset(index i) const
	{
		const data& dat = pa[i];
		return dat.is_int() && dat.get_int() <= 0 
			|| dat.is_atom() && strcmp(dat.get_atom(),"false") == 0
			|| dat.is_double() && dat.get_float() <= 0.0;
	}

} // namespace lp

#endif

